//
//  CMS_Database.swift
//  Composr Mobile SDK
//
//  Created by Aaswini on 06/08/14.
//  Copyright (c) 2014 Aaswini. All rights reserved.
//
/*
 
 CMS_Database (should work via SQLite which doesn't really use field types so we will ignore those)
 
 void add_table_field(string tableName, string fieldName)
 void rename_table_field(string tableName, string oldFieldName, string newFieldName)
 void delete_table_field(string tableName, string fieldName)
 void create_table(string tableName, array fieldNames)
 void drop_table_if_exists(string tableName)
 string db_escape_string(string value)
 array query(string query)
 void query_delete(string tableName, map whereMap)
 void query_insert(string tableName, map valueMap)
 array query_select(string tableName, array selectList, map whereMap, string extraSQL)
 string query_select_value(string tableName, string selectFieldName, map whereMap, string extraSQL) - returns blank if no match
 int query_select_int_value(string tableName, string selectFieldName, map whereMap, string extraSQL) - returns -1 if no match
 void query_update(string tableName, map valueMap, map whereMap)
 
 */

import UIKit
import Foundation

typealias CMSDatabaseUpgradeBlock = (dbInstance: sqlite3) -> Void

let k_DBVERSION = "db_version"
var objDBHandler: sqlite3? = nil

class CMS_Database: NSObject {
    
    /**
     *  Copy database file from the bundle to documents directory if DB doesn't exist.
     */
    class func createCopyOfDatabaseIfNeeded() {
        var paths = NSSearchPathForDirectoriesInDomains(.DocumentDirectory, .UserDomainMask, true)
        var documentsDirectory = paths[0]
        var writableDBPath = NSURL(fileURLWithPath: documentsDirectory).URLByAppendingPathComponent("cms_DB.sqlite").absoluteString
            // First, test for existence.
        var fileManager = NSFileManager.defaultManager()
        if NSProcessInfo.processInfo().arguments().indexOf("database_reset") != NSNotFound {
            do {
                try fileManager.removeItemAtPath(writableDBPath)
            }
            catch let error {
            }
        }
        if fileManager.fileExistsAtPath(writableDBPath) {
            return
        }
            // The writable database does not exist, so copy the default to the appropriate location.
        var defaultDBPath = NSURL(fileURLWithPath: NSBundle.mainBundle().resourcePath!).URLByAppendingPathComponent("cms_DB.sqlite").absoluteString
        var error: NSError?
        var success = try! fileManager.copyItemAtPath(defaultDBPath, toPath: writableDBPath)
        if !success {
            NSAssert1(0, "Failed to create writable database file with message '%@'.", error!.localizedDescription)
        }
    }

    /**
     *  Opens the db from the documents directory. Performs any upgrades if needed and keeps the db ready to work.
     */
    class func initializeDatabase() {
            // The database is stored in the application bundle.
            // Open the database. The database was prepared outside the application.
        var paths = NSSearchPathForDirectoriesInDomains(.DocumentDirectory, .UserDomainMask, true)
        var documentsDirectory = paths[0]
        var writableDBPath = NSURL(fileURLWithPath: documentsDirectory).URLByAppendingPathComponent("cms_DB.sqlite").absoluteString
        print("DB Path = \(writableDBPath)")
        if sqlite3_open(writableDBPath.UTF8String, objDBHandler) == SQLITE_OK {
            print("Database connection open successfully")
            //        [self upgradeDatabaseIfRequired:nil];
        }
        else {
            sqlite3_close(objDBHandler)
        }
    }

    /**
     *  Upgrade db if new version of app is released.
     */
    class func upgradeDatabaseIfRequired(upgradeBlock: CMSDatabaseUpgradeBlock) {
        if upgradeBlock {
            upgradeBlock(objDBHandler)
        }
        else {
            var defaults = NSUserDefaults.standardUserDefaults()
            var previousVersion = defaults.objectForKey(k_DBVERSION)!
            var currentVersion = self.versionNumberString()
            if previousVersion == nil {
                // do nothing. first install.
            }
            else if previousVersion.compare(currentVersion, options: .NumericSearch) == NSOrderedAscending {
                    // previous < current
                    // place upgrade codes here for each versions in their corresponding if structures.
                    // You can use the "objDBHandler" variable to access the db.
                var prevVersion = Int((previousVersion as NSString ?? "0").intValue)
                if prevVersion < 1 {
                    // place all codes for upgradation to version 1
                    prevVersion = 1
                }
                if prevVersion < 2 {
                    // place all codes for upgradation to version 2
                    prevVersion = 2
                }
            }

            defaults.setObject(currentVersion, forKey: k_DBVERSION)
            defaults.synchronize()
        }
    }

    /**
     *  Returns the version of the app
     *
     *  @return NSString version of the app from info.plist
     */
    class func versionNumberString() -> String {
        var infoDictionary = NSBundle.mainBundle().infoDictionary!
        var majorVersion = (infoDictionary["CFBundleShortVersionString"] as! String)
        return majorVersion
    }

    /**
     *  Add a new column to a table in db
     *
     *  @param tableName table name
     *  @param fieldName new field name
     */
    class func add_table_field(tableName: String, fieldName: String) {
        var sqlTmp = "ALTER TABLE \(tableName) ADD COLUMN \(fieldName) TEXT;" 
        let sqlStmt = sqlTmp.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            if sqlite3_step(cmp_sqlStmt) == SQLITE_DONE {
                print("Added new field")
            }
        }
        else {
            print("Error in adding field: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
    }

    /**
     *  Rename a column in the db
     *
     *  @param tableName    table name
     *  @param oldFieldName old column name
     *  @param newFieldName new column name
     */
    class func rename_table_field(tableName: String, oldFieldName: String, newFieldName: String) {
        var fieldNames = self.getFieldNamesForTable(tableName)
        var newFieldNames = [AnyObject]()
        for val: String in fieldNames {
            if (val == oldFieldName) {
                newFieldNames.append(newFieldName)
            }
            else {
                newFieldNames.append(val)
            }
        }
        var data = self.query_select(tableName, [""], [:], "")
        self.drop_table_if_exists(tableName)
        self.create_table(tableName, newFieldNames)
        for dict: [NSObject : AnyObject] in data {
            self.query_insert(tableName, dict)
        }
    }

    /**
     *  Delete a column from the db
     *
     *  @param tableName table name
     *  @param fieldName column name to delete
     *
     *  Warning : Data in the column will be truncated
     */
    class func delete_table_field(tableName: String, fieldName: String) {
        var fieldNames = self.getFieldNamesForTable(tableName)
        var newFieldNames = [AnyObject]()
        for val: String in fieldNames {
            if !(val == fieldName) {
                newFieldNames.append(val)
            }
        }
        var data = self.query_select(tableName, [""], [:], "")
        var newData = [AnyObject]()
        for dict: [NSObject : AnyObject] in data {
            var newDict = [NSObject : AnyObject]()
            for key: String in dict.allKeys() {
                if !(key == fieldName) {
                    newDict[key] = (dict[key] as! String)
                }
            }
        }
        self.drop_table_if_exists(tableName)
        self.create_table(tableName, newFieldNames)
        for dict: [NSObject : AnyObject] in newData {
            self.query_insert(tableName, dict)
        }
    }

    /**
     *  Create a new table with columns. All columns will be created with TEXT as type
     *
     *  @param tableName  table name
     *  @param fieldNames columns in the table
     */
    class func create_table(tableName: String, fieldNames: [AnyObject]) {
        var createQuery = "CREATE TABLE IF NOT EXISTS \(tableName)("
        for i in 0..<fieldNames.count {
            createQuery = createQuery.stringByAppendingString("%@ TEXT")
            if i != fieldNames.count - 1 {
                createQuery = createQuery.stringByAppendingString(" ,")
            }
        }
        createQuery = createQuery.stringByAppendingString(");")
        var sqlTmp = CMS_Strings.stringWithFormat(createQuery, array: fieldNames)
        let sqlStmt = sqlTmp.UTF8String
        var returnValue = sqlite3_exec(objDBHandler, sqlStmt, nil, nil, nil)
        if returnValue == SQLITE_OK {
            print("Created new table")
        }
        else {
            print("Error in creating table: \(sqlite3_errmsg(objDBHandler))")
        }
    }

    /**
     *  Deletes a table if exists
     *
     *  @param tableName table name to delete
     */
    class func drop_table_if_exists(tableName: String) {
        var sqlTmp = "DROP TABLE IF EXISTS \(tableName);" 
        let sqlStmt = sqlTmp.UTF8String
        var returnValue = sqlite3_exec(objDBHandler, sqlStmt, nil, nil, nil)
        if returnValue == SQLITE_OK {
            print("Deleted table")
        }
        else {
            print("Error in deleting table: \(sqlite3_errmsg(objDBHandler))")
        }
    }

    /**
     *  Checks if a table exists in the database with table name
     *
     *  @param tableName table name to check
     *
     *  @return YES if table exists else NO
     */
    class func isTableExists(tableName: String) -> Bool {
        if !tableName || (tableName == "") {
            return false
        }
        var query = "SELECT DISTINCT tbl_name FROM sqlite_master WHERE tbl_name = '\(tableName)'"
        let sqlStmt = query.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            if sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var value = NSString.stringWithUTF8String((sqlite3_column_text(cmp_sqlStmt, 0) as! CChar))
                if (value == tableName) {
                    sqlite3_finalize(cmp_sqlStmt)
                    return true
                }
            }
        }
        else {
            print("Error in selecting value: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return false
    }

    /**
     *  Escapes an string to suite to sqlite query format.
     *  Currently, only escapes single quotes.
     *
     *  @param value string to be escaped
     *
     *  @return escaped string
     */
    class func db_escape_string(value: String) -> String {
        return value.stringByReplacingOccurrencesOfString("'", withString: "''")
    }

    /**
     *  Execute a query and return the result set
     *
     *  @param query query to be executed
     *
     *  @return Array of dictionaries. Each dictionary corresponds to each row in the resultant table.
     */
    class func query(query: String) -> [AnyObject] {
        let sqlStmt = query.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            var result = [AnyObject]()
            while sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var row = [NSObject : AnyObject]()
                var numberOfColumns = sqlite3_data_count(cmp_sqlStmt)
                for i in 0..<numberOfColumns {
                    var columnName = NSString.stringWithUTF8String(sqlite3_column_name(cmp_sqlStmt, i))
                    var value = NSString.stringWithUTF8String((sqlite3_column_text(cmp_sqlStmt, i) as! CChar))
                    row[columnName] = value
                }
                result.append(row)
            }
            return result
        }
        else {
            print("Error in selecting rows: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return []
    }

    /**
     *  Delete a value from table with condition
     *
     *  @param tableName table name
     *  @param whereMap  condition
     */
    class func query_delete(tableName: String, whereMap: [NSObject : AnyObject]) {
        var deleteQuery = "DELETE FROM \(tableName) WHERE "
        deleteQuery = deleteQuery.stringByAppendingString(CMS_Database.convertWhereMapToSQL(whereMap, "AND"))
        deleteQuery = deleteQuery.stringByAppendingString(" ;")
        var sqlTmp = deleteQuery
        let sqlStmt = sqlTmp.UTF8String
        var returnValue = sqlite3_exec(objDBHandler, sqlStmt, nil, nil, nil)
        if returnValue == SQLITE_OK {
            print("Deleted row")
        }
        else {
            print("Error in deleting row: \(sqlite3_errmsg(objDBHandler))")
        }
    }

    /**
     *  Insert single record into table
     *
     *  @param tableName table name
     *  @param valueMap  record as a dictionary with column name mapped to value
     */
    class func query_insert(tableName: String, valueMap: [NSObject : AnyObject]) {
        var insertQuery = "INSERT INTO \(tableName) "
        insertQuery = insertQuery.stringByAppendingString(self.convertValueMapToInsertSQL(valueMap))
        insertQuery = insertQuery.stringByAppendingString(" ;")
        var sqlTmp = insertQuery
        let sqlStmt = sqlTmp.UTF8String
        var returnValue = sqlite3_exec(objDBHandler, sqlStmt, nil, nil, nil)
        if returnValue == SQLITE_OK {
            print("Inserted row")
        }
        else {
            print("Error in inserting row: \(sqlite3_errmsg(objDBHandler))")
        }
    }

    /**
     *  Select query
     *
     *  @param tableName  table name
     *  @param selectList list of columns to be selected
     *  @param whereMap   where conditions
     *  @param extraSQL   any extra sql codes
     *
     *  @return returns the result set as an array of dictionaries. Each dictionary corresponds to each row in the resultant table.
     */
    class func query_select(tableName: String, selectList: [AnyObject], whereMap: [NSObject : AnyObject], extraSQL: String) -> [AnyObject] {
        var selectQuery: String
        if whereMap {
            selectQuery = "SELECT \(CMS_Arrays.implode(",", selectList)) FROM \(tableName) WHERE \(self.convertWhereMapToSQL(whereMap, "AND")) "
        }
        else {
            selectQuery = "SELECT \(CMS_Arrays.implode(",", selectList)) FROM \(tableName) ;"
        }
        if extraSQL != nil {
            selectQuery = selectQuery.stringByAppendingString(extraSQL)
        }
        selectQuery = selectQuery.stringByAppendingString(" ;")
        var sqlTmp = selectQuery
        let sqlStmt = sqlTmp.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            var result = [AnyObject]()
            while sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var row = [NSObject : AnyObject]()
                var numberOfColumns = sqlite3_data_count(cmp_sqlStmt)
                for i in 0..<numberOfColumns {
                    var columnName = NSString.stringWithUTF8String(sqlite3_column_name(cmp_sqlStmt, i))
                    var value = NSString.stringWithUTF8String((sqlite3_column_text(cmp_sqlStmt, i) as! CChar))
                    row[columnName] = value
                }
                result.append(row)
            }
            return result
        }
        else {
            print("Error in selecting rows: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return []
    }

    /**
     *  Select single string from a table.
     *
     *  @param tableName       table name
     *  @param selectFieldName column name to select
     *  @param whereMap        where condition
     *  @param extraSQL        any extra sql codes if required
     *
     *  @return returns resultant single value as a string
     */
    class func query_select_value(tableName: String, selectFieldName: String, whereMap: [NSObject : AnyObject], extraSQL: String) -> String {
        var selectQuery = "SELECT \(selectFieldName) FROM \(tableName) WHERE \(self.convertWhereMapToSQL(whereMap, "AND")) "
        if extraSQL != nil {
            selectQuery = selectQuery.stringByAppendingString(extraSQL)
        }
        selectQuery = selectQuery.stringByAppendingString(" ;")
        var sqlTmp = selectQuery
        let sqlStmt = sqlTmp.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            if sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var value = NSString.stringWithUTF8String((sqlite3_column_text(cmp_sqlStmt, 0) as! CChar))
                sqlite3_finalize(cmp_sqlStmt)
                return value
            }
        }
        else {
            print("Error in selecting value: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return ""
    }

    /**
     *  Select single integer value from a table.
     *
     *  @param tableName       table name
     *  @param selectFieldName column name to select
     *  @param whereMap        where condition
     *  @param extraSQL        any extra sql codes if required
     *
     *  @return returns resultant single value as an int. -1 if none found.
     */
    class func query_select_int_value(tableName: String, selectFieldName: String, whereMap: [NSObject : AnyObject], extraSQL: String) -> Int {
        var selectQuery = "SELECT \(selectFieldName) FROM \(tableName) WHERE \(self.convertWhereMapToSQL(whereMap, "AND")) "
        if extraSQL != nil {
            selectQuery = selectQuery.stringByAppendingString(extraSQL)
        }
        selectQuery = selectQuery.stringByAppendingString(" ;")
        var sqlTmp = selectQuery
        let sqlStmt = sqlTmp.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            if sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var value = sqlite3_column_int(cmp_sqlStmt, 0)
                sqlite3_finalize(cmp_sqlStmt)
                return value
            }
        }
        else {
            print("Error in selecting value: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return -1
    }

    /**
     *  Simple update query
     *
     *  @param tableName table name
     *  @param valueMap  dictionary of values to be updated with keys as column name
     *  @param whereMap  where condition
     */
    class func query_update(tableName: String, valueMap: [NSObject : AnyObject], whereMap: [NSObject : AnyObject]) {
        var updateQuery = "UPDATE \(tableName) SET \(self.convertWhereMapToSQL(valueMap, ",")) WHERE \(self.convertWhereMapToSQL(whereMap, "AND")) "
        updateQuery = updateQuery.stringByAppendingString(" ;")
        var sqlTmp = updateQuery
        let sqlStmt = sqlTmp.UTF8String
        var returnValue = sqlite3_exec(objDBHandler, sqlStmt, nil, nil, nil)
        if returnValue == SQLITE_OK {
            print("Updated row")
        }
        else {
            print("Error in updating row: \(sqlite3_errmsg(objDBHandler))")
        }
    }

    /**
     *  Column names in a table
     *
     *  @param tableName table name
     *
     *  @return list of names of columns in a table
     */
    class func getFieldNamesForTable(tableName: String) -> [AnyObject] {
        var fieldsQuery = "PRAGMA table_info(\(tableName));"
        let sqlStmt = fieldsQuery.UTF8String
        var cmp_sqlStmt: sqlite3_stmt?
        var returnValue = sqlite3_prepare_v2(objDBHandler, sqlStmt, -1, cmp_sqlStmt, nil)
        if returnValue == SQLITE_OK {
            var result = [AnyObject]()
            while sqlite3_step(cmp_sqlStmt) == SQLITE_ROW {
                var value = NSString.stringWithUTF8String((sqlite3_column_text(cmp_sqlStmt, 1) as! CChar))
                result.append(value)
            }
            return result
        }
        else {
            print("Error in selecting rows: \(sqlite3_errmsg(objDBHandler))")
        }
        sqlite3_finalize(cmp_sqlStmt)
        return []
    }
    
    // converts where map in the format - key1='value1', key2='value2', ...
    class func convertWhereMapToSQL(whereMap: [NSObject : AnyObject], conjuction: String) -> String {
        var whereSQL = ""
        for i in 0..<whereMap.allKeys().count {
            whereSQL = whereSQL.stringByAppendingString("\(whereMap.allKeys()[i]) = ")
            whereSQL = whereSQL.stringByAppendingString("'%@'")
            if i != whereMap.allKeys().count - 1 {
                whereSQL = whereSQL.stringByAppendingFormat(" %@ ", conjuction)
            }
        }
        return CMS_Strings.stringWithFormat(whereSQL, array: CMS_Arrays.Array_values(false, whereMap))
    }
    
    // converts value map in the format - (key1 , key2, key3, ..) VALUES ('value1', 'value2', 'value3', ...)
    class func convertValueMapToInsertSQL(valueMap: [NSObject : AnyObject]) -> String {
        var insertSQL = String() /* capacity: 10 */
        insertSQL += "("
        var keys = valueMap.allKeys()
        for i in 0..<keys.count {
            insertSQL += "\(keys[i])"
            if i != keys.count - 1 {
                insertSQL += ", "
            }
        }
        insertSQL += ") VALUES ("
        for i in 0..<keys.count {
            insertSQL += "'\(valueMap[keys[i]] as! String)'"
            if i != keys.count - 1 {
                insertSQL += ", "
            }
        }
        insertSQL += ")"
        return insertSQL
    }
}